/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/* THIS EMULATION IS INCOMPLETE. MORE TESTING IS NEEDED ON A REAL DEVICE */

#include "cpc.h"
#include "emudevice.h"

//https://www.mirrorservice.org/sites/www.bitsavers.org/pdf/westernDigital/WD100x/79-000029_WD1002S-WX2_XT_MFM_OEM_Manual_Jul85.pdf

#define HD20_CMD_TEST_DRIVE_READY 0x0
#define HD20_CMD_RECALIBRATE 0x01
#define HD20_CMD_READ_STATUS 0x03
#define HD20_CMD_FORMAT 0x04
#define HD20_CMD_VERIFY_SECTORS 0x05
#define HD20_CMD_FORMAT_TRACK 0x06
#define HD20_CMD_FORMAT_BAD_TRACK 0x06
#define HD20_CMD_READ_SECTORS 0x08
#define HD20_CMD_WRITE_SECTORS 0x0a
#define HD20_CMD_SEEK 0x0b
#define HD20_CMD_INITIALIZE_DRIVE_PARAMS 0x0c
#define HD20_CMD_READ_ECC_BURST_ERROR_LENGTH 0x0d
#define HD20_CMD_READ_SECTOR_BUFFER 0x0e
#define HD20_CMD_WRITE_SECTOR_BUFFER 0x0f
#define HD20_CMD_EXECUTE_SECTOR_BUFFER_DIAGNOSTIC 0x0e0
#define HD20_CMD_EXECUTE_DRIVE_DIAGNOSTIC 0x0e3
#define HD20_CMD_EXECUTE_CONTROLLER_DIAGNOSTIC 0x0e4
#define HD20_CMD_READ_LONG 0x0e5
#define HD20_CMD_WRITE_LONG 0x0e6


/* appears to be based on wd1002 winchester controller */
/* Multi sector operation? */
enum
{
	STATE_RESET,
	STATE_IDLE,
	STATE_SELECTION,
	STATE_COMMAND_FIRST_BYTE,
	STATE_COMMAND_BYTES,
	STATE_DATA,
	STATE_STATUS
};

/* Bits for command status byte */
#define CSB_ERROR       0x02
#define CSB_LUN 		0x20

/* status bits */
#define HD20_STA_REQ	0x01	/* ready for transfer (1=requesting) */
#define HD20_STA_IO		0x02	/* device->host (1=input) */
#define HD20_STA_CD 	0x04   /* control/status (1=data) */
#define HD20_STA_BUSY	0x08
#define HD20_STA_DRQ 	0x10
#define HD20_STA_IRQ	0x20

#define HD20_STA_REQUEST 0x01
#define HD20_STA_INPUT 0x02
#define HD20_STA_OUTPUT 0x00
#define HD20_STA_DATA 0x04
#define HD20_STA_CONTROL 0x00
#define HD20_STA_BUSY 0x08


#define CTL_DRQEN 		0x01	/* 1=DMA Request enabled */
#define CTL_IRQEN 		0x02	/* 1=Enables interrupst to the host */

typedef struct
{
	int SystemROMIndex;
	int ROMIndex;
	BOOL Ext;
	BOOL LK464;
	unsigned char *HardDisc;

	unsigned char SystemRom[16384];
} DobbertinHD20;

static DobbertinHD20 dob;

void dobbertin_hd20_SetROM(const unsigned char *pROM, unsigned long RomLength)
{
	EmuDevice_CopyRomData(dob.SystemRom, sizeof(dob.SystemRom), pROM, RomLength);
}


void dobbertin_hd20_ClearROM(void)
{
	EmuDevice_ClearRomData(dob.SystemRom, sizeof(dob.SystemRom));
}


void dobbertin_hd20_MemoryRethink(MemoryData *pData)
{
    if (dob.SystemROMIndex==dob.ROMIndex)
    {
        /* read only */
        pData->pReadPtr[7] = dob.SystemRom;
        pData->pReadPtr[6] = dob.SystemRom;
		pData->pReadMaskPtr[7] = GetDefaultReadMask() - 0x0c000;
		pData->pReadMaskPtr[6] = GetDefaultReadMask() - 0x0c000;
		pData->bRomDisable[7] = TRUE;
		pData->bRomDisable[6] = TRUE;
    }
}

typedef struct  
{
	int State;


	int nBytesRemaining;
	int CommandBuffer[ 16 ];
	int NumCylinders;
	int NumHeads;
	int CurrentSector;
	int CurrentCylinder;
	int CurrentHead;
	int BlockCount;
	/* 256, 512 or 1024 bytes */
	int BlockSize;
	/* 32,18,17 or 9 */
	int SectorsPerTrack;

	unsigned char Buffer[ 1024 ];
	int nBufferPos;

	int Result;
	int Status;
	int Configuration;
} DobbertinHardDiskController ;

static DobbertinHardDiskController controller;

void enter_command_state(void)
{
	controller.State = STATE_COMMAND_FIRST_BYTE;
	controller.Status |= HD20_STA_CONTROL | HD20_STA_REQUEST;
}


void enter_idle_state(void)
{
	/* enter idle state */
	controller.State = STATE_IDLE;
}


void enter_data_state(int nBytes, int nInput)
{
	/* position in buffer */
	controller.nBufferPos = 0;
	/* number of bytes */
	controller.nBytesRemaining = nBytes;
	/* data state */
	controller.State = STATE_DATA;
	if (nInput)
	{
		/* read */
		controller.Status |= HD20_STA_INPUT;
	}
	controller.Status &= ~HD20_STA_CONTROL;
	/* request data */
	controller.Status |= HD20_STA_REQUEST;
}

unsigned long get_sector_offset(void)
{
	unsigned long SectorOffset = ( controller.CurrentCylinder*controller.SectorsPerTrack*controller.NumHeads ) + ( controller.SectorsPerTrack * controller.CurrentHead ) + controller.CurrentSector;
	return SectorOffset;
}


void enter_result_state(void)
{
	controller.Status = HD20_STA_REQUEST | HD20_STA_BUSY | HD20_STA_INPUT | HD20_STA_CONTROL;
	controller.State = STATE_STATUS;
}

void transfer_sector_start(int read)
{
	if (read)
	{
		unsigned long SectorOffset = get_sector_offset();
		unsigned long Offset = SectorOffset*controller.BlockSize;
		memcpy( controller.Buffer, &dob.HardDisc[ Offset ], controller.BlockSize );
	}

	enter_data_state( controller.BlockSize, read );
	controller.nBytesRemaining = controller.BlockSize;
}

void transfer_sector(int read)
{

	if (!read)
	{
		unsigned long SectorOffset = get_sector_offset();
		unsigned long Offset = SectorOffset*controller.BlockSize;
		memcpy( &dob.HardDisc[ Offset ], controller.Buffer, controller.BlockSize );
	}

	controller.BlockCount--;
	if ( controller.BlockCount != 0 )
	{
		controller.CurrentSector++;
		if ( controller.CurrentSector == controller.SectorsPerTrack )
		{
			controller.CurrentSector = 0;
			controller.CurrentHead++;
			if ( controller.CurrentHead == controller.NumHeads )
			{
				controller.CurrentHead = 0;
				controller.CurrentCylinder++;
			}
		}

		transfer_sector_start(read);
	}
	else
	{
		enter_result_state();

	}
}

void enter_reset_state(void)
{
	controller.State = STATE_RESET;
	controller.Status = 0;

	enter_idle_state();
}


BOOL dobbertin_hd20_read(Z80_WORD Addr, Z80_BYTE *pDeviceData)
{
	switch (Addr & 0x03)
    {
		/* data read */
		case 0:
		{
			Z80_BYTE Data=0x0ff;
			if ( controller.State == STATE_STATUS )
			{
				Data = controller.Result;

				controller.Status &= ~( HD20_STA_CONTROL | HD20_STA_INPUT | HD20_STA_BUSY | HD20_STA_REQUEST );
				enter_idle_state();
			}
			else if ( controller.State == STATE_DATA )
			{
				Data = controller.Buffer[ controller.nBufferPos ];
				controller.nBufferPos++;
				controller.nBytesRemaining--;
				if ( controller.nBytesRemaining == 0 )
				{
					if ( controller.CommandBuffer[ 0 ] == HD20_CMD_READ_STATUS)
					{
						enter_result_state();
					}
					else
						if ( controller.CommandBuffer[ 0 ] == HD20_CMD_READ_SECTORS )
					{
						transfer_sector(1);
					}
					else
					{
						enter_result_state();
					}
				}
			}
			*pDeviceData = Data;
		}
		return TRUE;

		/* status */
        case 1:
			*pDeviceData = controller.Status;
			return TRUE;

		/* configuration */
        case 2:
			*pDeviceData = controller.Configuration;
			return TRUE;
		default:
			break;
    }

    return FALSE;
}

BOOL dobbertin_hd20_reset_r(Z80_WORD Addr, Z80_BYTE *Data)
{
	return FALSE;
}


void dobbertin_hd20_reset_w(Z80_WORD Addr, Z80_BYTE Data)
{
}

void dobbertin_hd20_write(Z80_WORD Addr, Z80_BYTE Data)
{
	switch (Addr & 0x03)
    {
		/* data write */
	case 0:
	{
		if ( controller.State == STATE_DATA )
		{
			controller.Status &= ~HD20_STA_REQUEST;
			controller.Buffer[ controller.nBufferPos ] = Data;
			controller.nBufferPos++;
			controller.nBytesRemaining--;
			if ( controller.nBytesRemaining != 0 )
			{
				controller.Status |= HD20_STA_REQUEST;

			}
			else
			{
				if ( controller.CommandBuffer[ 0 ] == HD20_CMD_INITIALIZE_DRIVE_PARAMS )
				{
					controller.NumCylinders = ( ( controller.Buffer[ 0 ] & 0x0ff ) << 8 ) | ( controller.Buffer[ 1 ] & 0x0ff );
					controller.NumHeads = controller.Buffer[ 2 ] & 0x0ff;
				//	printf("Num cylinders: %d\n", NumCylinders);
				//	printf("Num heads: %d\n", NumHeads);
				
				}

				if ( controller.CommandBuffer[ 0 ] == HD20_CMD_WRITE_SECTORS )
				{
					transfer_sector(0);
				}
				else
				{
					enter_result_state();
				}
			}
		}
		else
			if ( controller.State == STATE_COMMAND_BYTES )
		{
			controller.Status &= ~HD20_STA_REQUEST;

			controller.CommandBuffer[ controller.nBufferPos ] = Data;
			controller.nBufferPos++;
			controller.nBytesRemaining--;

			if ( controller.nBytesRemaining != 0 )
			{
				controller.Status |= HD20_STA_REQUEST;
			}
			else
			{
				controller.Status &= ~HD20_STA_CONTROL;
				if ( controller.CommandBuffer[ 0 ] == HD20_CMD_INITIALIZE_DRIVE_PARAMS)
				{
					/* host write 8 bytes */
					enter_data_state(8, 0);
				}
				else if ( controller.CommandBuffer[ 0 ] == HD20_CMD_TEST_DRIVE_READY)
				{
					enter_result_state();
				}
				else if ( controller.CommandBuffer[ 0 ] == HD20_CMD_READ_STATUS )
				{
					enter_data_state(4,1);
				}
				else if ( controller.CommandBuffer[ 0 ] == HD20_CMD_SEEK )
				{
					enter_result_state();
				}
				else if ( controller.CommandBuffer[ 0 ] == HD20_CMD_READ_SECTORS )
				{
					controller.CurrentSector = controller.CommandBuffer[ 2 ] & 0x03f;
					controller.CurrentCylinder = ( controller.CommandBuffer[ 3 ] & 0x0ff ) | ( ( controller.CommandBuffer[ 2 ] << 2 ) & 0x0300 );
					controller.CurrentHead = controller.CommandBuffer[ 1 ] & 0x01f;
					controller.BlockCount = controller.CommandBuffer[ 4 ] & 0x0ff;
					if ( controller.BlockCount == 0 )
					{
						controller.BlockCount = 256;
					}
				//	printf("Sector: %d\n", CurrentSector);
				//	printf("Current Cylinder: %d\n", CurrentCylinder);
				//	printf("Current Head: %d\n", CurrentHead);
				//	printf("Num blocks: %d\n", BlockCount);
					transfer_sector_start(1);
				}
				else if ( controller.CommandBuffer[ 0 ] == HD20_CMD_WRITE_SECTORS )
				{
					controller.CurrentSector = controller.CommandBuffer[ 2 ] & 0x03f;
					controller.CurrentCylinder = ( controller.CommandBuffer[ 3 ] & 0x0ff ) | ( ( controller.CommandBuffer[ 2 ] << 2 ) & 0x0300 );
					controller.CurrentHead = controller.CommandBuffer[ 1 ] & 0x01f;
					controller.BlockCount = controller.CommandBuffer[ 4 ] & 0x0ff;
					if ( controller.BlockCount == 0 )
					{
						controller.BlockCount = 256;
					}
				//	printf("Sector: %d\n", CurrentSector);
				//	printf("Current Cylinder: %d\n", CurrentCylinder);
				//	printf("Current Head: %d\n", CurrentHead);
				//	printf("Num blocks: %d\n", BlockCount);
					transfer_sector_start(0);
				}
			}
		}
		else
			if ( controller.State == STATE_COMMAND_FIRST_BYTE )
		{
			controller.Status &= ~HD20_STA_REQUEST;

			/* first command byte */
			controller.CommandBuffer[ 0 ] = Data;
			controller.nBufferPos = 1;
			switch (Data)
			{
			case HD20_CMD_INITIALIZE_DRIVE_PARAMS:
			{
				controller.nBytesRemaining = 6 - 1;
				controller.State = STATE_COMMAND_BYTES;
				controller.Status |= HD20_STA_REQUEST;
			}
			break;

			case HD20_CMD_TEST_DRIVE_READY:
			{
				controller.nBytesRemaining = 6 - 1;
				controller.State = STATE_COMMAND_BYTES;
				controller.Status |= HD20_STA_REQUEST;
			}
			break;

			case HD20_CMD_READ_SECTORS:
			{
				controller.nBytesRemaining = 6 - 1;
				controller.State = STATE_COMMAND_BYTES;
				controller.Status |= HD20_STA_REQUEST;
			}
			break;
			case HD20_CMD_WRITE_SECTORS:
			{
				controller.nBytesRemaining = 6 - 1;
				controller.State = STATE_COMMAND_BYTES;
				controller.Status |= HD20_STA_REQUEST;
			}
			break;
			case HD20_CMD_SEEK:
			{
				controller.nBytesRemaining = 6 - 1;
				controller.State = STATE_COMMAND_BYTES;
				controller.Status |= HD20_STA_REQUEST;
			}
			break;


			default:
				break;
			}
		}
	}
	break;
        case 1:
        {
			/* reset */
			enter_reset_state();
        }
        break;

		/* select */
        case 2:
        {
			/* only accepted in idle state */
			if ( controller.State != STATE_IDLE )
				break;

			/* set busy and enter command state */
			controller.Status |= HD20_STA_BUSY;

			/* enter command state */
			enter_command_state();
		}
        break;

		default:
			break;
    }




}

void dobbertin_hd20_reset(Z80_WORD Addr, Z80_BYTE Data)
{
}


static CPCPortRead DobbertinRead[2]=
{
	/* A15-A10, A8-A7 decoded by U11 */
	/* A4,A3 from U9B */
	/* A9 is G1, A2 = A, A5 = B, A6 = C into U10 */
	/* A1,A0 are registers */
	/* U11 decodes as pin 12 */
	/* ‭11111011111000rr‬ - according to U11 */
	{
		0x0fbe0,
		0x0fbe0,
		dobbertin_hd20_read
	},
	/* ‭11111011111001xx - according to U11 */
	{
		0x0fbe4,
		0x0fbe4,
		dobbertin_hd20_reset_r
	},
		/* there is another I/O port that doesn't seem to make sense */

};

void dobbertin_hd20_rom_write(Z80_WORD Addr, Z80_BYTE Data)
{
	/* U6 provides the decoding for the ROM, SV1 is the selection */
	/* U6  shows that all 8 bits are fully decoded, but it only decodes 7 values */
	dob.SystemROMIndex = Data;
	
	Computer_RethinkMemory();
}


static EmuDeviceRom DobbertinROM[1]=
{
	{
	"Dobbertin HD20 System ROM",
	"SystemRom",
	dobbertin_hd20_SetROM,
	dobbertin_hd20_ClearROM,
	false,
	false,
	sizeof(dob.SystemRom),
	0   /* ROM CRC - todo */
	}
};


#if 0
74ls138b chooses?

df00

fb80,f980
111110x11xxxxxxx

A15,A14,A13,A12,A11=1,/A10=0, A8,A7

/o19 = /IORQ * /M1		- interrupt acknowledge
	+
       ROMDIS * /ROMEN - romdisabled, but not our rom?
	+
       RAMDIS * /RAMRD - ramdisabled, but not reading?
	+
       /IORQ * /RD * M1 * A15 * A14 * A12 * A11 * A10    11x111 dfff
	+
       /IORQ * /RD * M1 * A15 * A14 * A12 * A11 * /A10 * /A8 (11x110x0xxxxxxxx)  d800			faff ‭1111101001111110‬
	+				
       /464 * /IORQ * /RD * M1 * A15 * A14 * A12 * A11 * /A10 * A8 * A7 (11x110x11xxxxxxx d980		fb7f is not ok.
	+

	464 * /IORQ * /RD * M1 * A15 * A14 * A12 * A11 * /A10 * A8 (11x110x1xxxxxxxx) d900		fb7f is ok (link closed)
#endif
   	

static CPCPortWrite DobbertinWrite[3]=
{
	/* A15-A10, A8-A7 decoded by U11 */
	/* A4,A3 from U9B */
	/* A9 is G1, A2 = A, A5 = B, A6 = C into U10 */
	/* A1,A0 are registers */
	/* U11 decodes as */
	/* ‭11111011111000rr‬ - according to U11 */
	{
		0x0fbe0,
		0x0fbe0,
		dobbertin_hd20_write
	},
	/* ‭11111011111001xx - according to U11 */
	{
		0x0fbe4,
		0x0fbe4,
		dobbertin_hd20_reset_w
	},
	/* System ROM Select */
	{
		0x02000,	/* confirmed by schematics */
		0x00000,
		dobbertin_hd20_rom_write
	}
};

void DobbertinHD20Device_Init(void)
{
	dob.Ext = TRUE;
	dob.LK464 = FALSE;
	dob.SystemROMIndex = 6;

	dob.HardDisc = (unsigned char *)malloc(20 * 1024 * 1024);
	memset(dob.HardDisc, 0xe5, 20 * 1024 * 1024);
}
void DobbertinHD20Device_Exit(void)
{
	free(dob.HardDisc);
}

static EmuDeviceMedia DobbertinHD20Media[1]=
{
	{
		CPC_MEDIA_TYPE_HARDDISK,
		0,
		"Hard disk",
		TRUE
	}
};

static EmuDevice DobbertinHD20Device =
{
	NULL,
	DobbertinHD20Device_Init,
	DobbertinHD20Device_Exit,
	"DOBHD20",
	"DobbertinHD20",
	"Dobbertin HD20 Hard disk",
	CONNECTION_EXPANSION,   /* connected to expansion */
	0,
	sizeof(DobbertinRead)/sizeof(DobbertinRead[0]),                /* 2 read port */
	DobbertinRead,
	sizeof(DobbertinWrite) / sizeof(DobbertinWrite[0]),                /* 3 write port */
	DobbertinWrite,
	0,                /* no memory read*/
	NULL,
	0,                /* no memory write */
	NULL,
	NULL, /* reset function */
	dobbertin_hd20_MemoryRethink, /* memory rethink */
	NULL, /* power function*/
	0,      /* no switches*/
	NULL,
	0,                      /* no buttons */
	NULL,
	sizeof(DobbertinHD20Media) / sizeof(DobbertinHD20Media[0]),
	DobbertinHD20Media,
	sizeof(DobbertinROM) / sizeof(DobbertinROM[0]),
	DobbertinROM,
	NULL,                   /* no cursor function */
	NULL,                 /* no generic roms */
	NULL,
	NULL,
	0,
	NULL,
	NULL, /* sound */
	NULL, /* lpen */
	NULL, /* reti */
	NULL, /* ack maskable interrupt */
	NULL, /* dkram data */
	NULL, /* device ram */
	NULL, /* device backup */
	NULL,
};

void DobbertinHD20_Init(void)
{
	memset( &controller, 0, sizeof( DobbertinHardDiskController ) );
	controller.State = STATE_RESET;
	/* 256, 512 or 1024 bytes */
	controller.BlockSize = 512;
	/* 32,18,17 or 9 */
	controller.SectorsPerTrack = 17;
	controller.Configuration = 1;

	RegisterDevice(&DobbertinHD20Device);
}

